/*
----------------------------------------------------------------------------

 Copyright (C) Sartorius Stedim Data Analytics AB 2017 -

 Use, modification and distribution are subject to the Boost Software
 License, Version 1.0. (See http://www.boost.org/LICENSE_1_0.txt)
----------------------------------------------------------------------------
*/

#include "stdafx.h"
#include "utf8util.h"
#include <string.h>
#include <stdlib.h>
#include "sqrunner.h"

void SQRunner_PrintModelInfo(SQ_Model hModel);
void SQRunner_Predict(SQRunner* pObj, SQPredictionset* oSQPredSet);
void SQRunner_PrintPredictionResults(SQ_Prediction hPrediction);
void CreateFakePredictionset(SQ_Model hModel, SQPredictionset* oSQPredSet);
void ClearPredictionset(SQPredictionset* pSQPredSet);
void GetBiPlot(SQ_Model hModel);
void GetContributions(SQ_Model hModel);

/* Function to print a string vector */
void SQRunner_PrintStringVector(SQ_StringVector pStringVector, char szSeparator)
{
   char strVal[1000];
   int iIter;
   int iNumStrings = 0;

   if (SQ_GetNumStringsInVector(pStringVector, &iNumStrings) != SQ_E_OK)
   {
      PrintUTF8String(stdout, "SQ_GetNumStringsInVector failed.");
      return;
   }

   for (iIter = 1; iIter <= iNumStrings; ++iIter)
   {
      /* Print the results */
      if (SQ_GetStringFromVector(pStringVector, iIter, strVal, sizeof(strVal)) != SQ_E_OK)
      {
         PrintUTF8String(stdout, "SQ_GetStringFromVector failed.");
         return;
      }
      UTF8_printf(stdout, "%s%c", strVal, szSeparator);
   }
   PrintUTF8String(stdout, "\n");
}

/* Function to print the vector data */
void SQRunner_PrintVectorData(SQ_VectorData pVectorData)
{
   SQ_FloatMatrix pMatrix = 0;
   SQ_StringVector pRowNames = 0;
   SQ_StringVector pColumnNames = 0;
   float fVal;
   int iColIter;
   int iRowIter;
   int iRows = 0;
   int iCols = 0;
   char strVal[1000];

   if (SQ_GetDataMatrix(pVectorData, &pMatrix) != SQ_E_OK)
   {
      PrintUTF8String(stdout, "SQ_GetDataMatrix failed.");
      return;
   }

   if (SQ_GetNumColumnsInFloatMatrix(pMatrix, &iCols) != SQ_E_OK)
   {
      PrintUTF8String(stdout, "SQ_GetNumColumnsInFloatMatrix failed.");
      return;
   }
   if (SQ_GetNumRowsInFloatMatrix(pMatrix, &iRows) != SQ_E_OK)
   {
      PrintUTF8String(stdout, "SQ_GetNumRowsInFloatMatrix failed.");
      return;
   }
   if (SQ_GetRowNames(pVectorData, &pRowNames) != SQ_E_OK)
   {
      PrintUTF8String(stdout, "SQ_GetRowNames failed.");
      return;
   }
   if (SQ_GetColumnNames(pVectorData, &pColumnNames) != SQ_E_OK)
   {
      PrintUTF8String(stdout, "SQ_GetColumnNames failed.");
      return;
   }

   SQRunner_PrintStringVector(pColumnNames, '\t');
   for (iRowIter = 0; iRowIter < iRows; ++iRowIter)
   {
      /* Print the row name */
      if (SQ_GetStringFromVector(pRowNames, iRowIter + 1, strVal, 1000) != SQ_E_OK)
      {
         PrintUTF8String(stdout, "SQ_GetStringFromVector failed.");
         return;
      }

      UTF8_printf(stdout, "%s\t", strVal);

      for (iColIter = 0; iColIter < iCols; ++iColIter)
      {
         /* Print the results tab separated to the file */
         if (SQ_GetDataFromFloatMatrix(pMatrix, iRowIter + 1, iColIter + 1, &fVal) != SQ_E_OK)
         {
            PrintUTF8String(stdout, "SQ_GetDataFromFloatMatrix failed.");
            return;
         }
         UTF8_printf(stdout, "%.8f\t", fVal);
      }
      PrintUTF8String(stdout, "\n");
   }
   PrintUTF8String(stdout, "\n");
}

void SQRunner_PrintProjectInfo(SQRunner* pObj)
{
   UTF8_printf(stdout, "Path to usp file:        %s\n", pObj->mszUSPName);
}

/* Print some information about the model */
void SQRunner_PrintModelInfo(SQ_Model hModel)
{
   char szBuffer[256];
   int iNumComponents;

   /* Model name */
   SQ_GetModelName(hModel, szBuffer, sizeof(szBuffer));

   UTF8_printf(stdout, "Model name: %s\n", szBuffer);

   /* Model type */
   SQ_GetModelTypeString(hModel, szBuffer, sizeof(szBuffer));
   UTF8_printf(stdout, "Model type: %s\n", szBuffer);

   /* Number of components */
   SQ_GetNumberOfComponents(hModel, &iNumComponents);
   UTF8_printf(stdout, "Number of components: %d\n", iNumComponents);
}


void SQRunner_Init(SQRunner* pObj, char* szUSPName)
{
   pObj->mszUSPName = strdup(szUSPName);

   /* Init handles to null */
   pObj->mProjectHandle = NULL;
   pObj->mModelHandle = NULL;
   pObj->mPredictionHandle = NULL;
   pObj->mbAllowPythonPlugin = 0;
}

void SQRunner_Destroy(SQRunner* pObj)
{
   SQ_CloseProject(&pObj->mProjectHandle); /* This will close all models and predictions as well */
   pObj->mProjectHandle = NULL;
}

void SQRunner_Run(SQRunner* pObj)
{
   int iModelNumber;
   SQ_Bool bValid;
   SQPredictionset oPredictionSet;

   /* If the license file has an OEM password in it we must specify it here. */
   /* SQ_SetOEMPassword("OEM_Password_For_My_Application"); */

   if (SQ_IsLicenseFileValid(&bValid) != SQ_E_OK)
   {
      PrintUTF8String(stdout, "Could not read license.");
      return;
   }
   if (!bValid)
   {
      PrintUTF8String(stdout, "Invalid license.");
      return;
   }

   /* This is the file we want to use in SIMCA-QP */
   SQRunner_PrintProjectInfo(pObj);

   /* Open the project */
   if (SQ_OpenProject(pObj->mszUSPName, NULL /* Not a password protected project*/, &pObj->mProjectHandle) != SQ_E_OK)
   {
      PrintUTF8String(stdout, "Could not open project. SQ_OpenProject failed.");
      return;
   }

   // SQ_EnableEmbeddedScripts must be called in order to allow running python code in usp
   if (pObj->mbAllowPythonPlugin > 0)
      SQ_EnableEmbeddedScripts(pObj->mProjectHandle);

   /* Open the first model */
   SQ_GetModelNumberFromIndex(pObj->mProjectHandle, 1, &iModelNumber);
   if (SQ_GetModel(pObj->mProjectHandle, iModelNumber, &pObj->mModelHandle) != SQ_E_OK)
   {
      PrintUTF8String(stdout, "Could not get the first model. SQ_GetModel failed.");
      return;
   }

   SQRunner_PrintModelInfo(pObj->mModelHandle);

   /* Get the data for the default BiPlot used in SIMCA. */
   /*GetBiPlot(pObj->mModelHandle); */

   /* Get the score contributions for the model. */
   /* GetContributions(pObj->mModelHandle); */

   /* Create a fake prediction set. In a real example, prediction vales should be taken from real data */
   CreateFakePredictionset(pObj->mModelHandle, &oPredictionSet);

   /* Do a prediction */
   SQRunner_Predict(pObj, &oPredictionSet);

   /* Clear the prediction set */
   ClearPredictionset(&oPredictionSet);

   /* Print the results */
   if (pObj->mPredictionHandle != NULL)
   {
      SQRunner_PrintPredictionResults(pObj->mPredictionHandle);
      SQ_ClearPrediction(&pObj->mPredictionHandle);
      pObj->mPredictionHandle = NULL;
   }
   else
      printf("Failed to do predictions.\n");
}

/* The actual prediction is done in this function */
void SQRunner_Predict(SQRunner* pObj, SQPredictionset* oSQPredSet)
{
   SQ_PreparePrediction oPreparePrediction = NULL;
   SQ_VariableVector oVariableVector = NULL;
   int iNumVariables = 0;
   SQ_Variable oVariable = NULL;
   int iVariable;
   int iRow;
   SQ_Bool bIsQualitative;
   int iQualPos = 0;
   int iQuantPos = 0;

   SQ_GetPreparePrediction(pObj->mModelHandle, &oPreparePrediction);
   SQ_GetVariablesForPrediction(oPreparePrediction, &oVariableVector);
   SQ_GetNumVariablesInVector(oVariableVector, &iNumVariables);

   /* Test if we've got correct number of prediction data */
   if (oSQPredSet->miNumQualCols + oSQPredSet->miNumQuantCols != iNumVariables)
   {
      PrintUTF8String(stdout, "Invalid number of variables to predict.\n");
      return;
   }

   /* Set the prediction data */
   for (iVariable = 0; iVariable < iNumVariables; ++iVariable)
   {
      SQ_GetVariableFromVector(oVariableVector, iVariable + 1, &oVariable);
      SQ_IsQualitative(oVariable, &bIsQualitative);
      for (iRow = 0; iRow < oSQPredSet->miNumRows; ++iRow)
      {
         if (bIsQualitative == SQ_True)
            SQ_SetQualitativeData(oPreparePrediction, iRow + 1, iVariable + 1, oSQPredSet->mpfQualitativeValues[iQualPos + oSQPredSet->miNumQualCols * iRow]);
         else
            SQ_SetQuantitativeData(oPreparePrediction, iRow + 1, iVariable + 1, oSQPredSet->mpfQuantitativeValues[iQuantPos + oSQPredSet->miNumQuantCols * iRow]);
      }

      if (bIsQualitative == SQ_True)
         ++iQualPos;
      else
         ++iQuantPos;
   }

   /* Get the prediction */
   SQ_GetPrediction(oPreparePrediction, &pObj->mPredictionHandle);
   SQ_ClearPreparePrediction(&oPreparePrediction);
}

/* Function to print a few PS vectors from the prediction */
void SQRunner_PrintPredictionResults(SQ_Prediction hPrediction)
{
   SQ_VectorData hVectorData = NULL;

   /* Get some vectors and print them */
   SQ_GetTPS(hPrediction, NULL, &hVectorData);
   SQRunner_PrintVectorData(hVectorData);

   SQ_GetDModXPS(hPrediction, NULL, SQ_Normalized_Default, SQ_ModelingPowerWeighted_Default, &hVectorData);
   SQRunner_PrintVectorData(hVectorData);

   SQ_ClearVectorData(&hVectorData);
}

/* Function to create a fake prediction set. */
void CreateFakePredictionset(SQ_Model hModel, SQPredictionset* oSQPredSet)
{
   SQ_PreparePrediction oPreparePrediction = NULL;
   SQ_VariableVector oVariableVector = NULL;
   SQ_Variable oVariable = NULL;
   SQ_StringVector oStringVec = NULL;
   int iNumVariables;
   int iNumQualitativeVariables = 0;
   int iVariable;
   int iNumberOfRows = 1;
   int iRow;
   float fVal = 0;
   SQ_Bool bIsQualitative;
   float* pfQuantPos;
   char** pszQualPos;
   char szBuffer[256];

   /* Get number of variables needed for prediction */
   SQ_GetPreparePrediction(hModel, &oPreparePrediction);
   SQ_GetVariablesForPrediction(oPreparePrediction, &oVariableVector);
   SQ_GetNumVariablesInVector(oVariableVector, &iNumVariables);

   /* Count number of qualitative variables */
   for (iVariable = 1; iVariable <= iNumVariables; ++iVariable)
   {
      SQ_GetVariableFromVector(oVariableVector, iVariable, &oVariable);
      SQ_IsQualitative(oVariable, &bIsQualitative);
      if (bIsQualitative == SQ_True)
         ++iNumQualitativeVariables;
   }

   /* Create a prediction set */
   oSQPredSet->miNumQualCols = iNumQualitativeVariables;
   oSQPredSet->miNumQuantCols = iNumVariables - iNumQualitativeVariables;
   oSQPredSet->miNumRows = iNumberOfRows;

   if (oSQPredSet->miNumQualCols > 0)
      oSQPredSet->mpfQualitativeValues = malloc(oSQPredSet->miNumQualCols * sizeof(char*) * iNumberOfRows);
   else
      oSQPredSet->mpfQualitativeValues = NULL;

   if (oSQPredSet->miNumQuantCols > 0)
      oSQPredSet->mpfQuantitativeValues = malloc(oSQPredSet->miNumQuantCols * sizeof(float) * iNumberOfRows);
   else
      oSQPredSet->mpfQuantitativeValues = NULL;

   /* Set fake values */
   pfQuantPos = oSQPredSet->mpfQuantitativeValues;
   pszQualPos = oSQPredSet->mpfQualitativeValues;

   for (iVariable = 0; iVariable < iNumVariables; ++iVariable)
   {
      SQ_GetVariableFromVector(oVariableVector, iVariable + 1, &oVariable); /* Index is 1 based */
      SQ_IsQualitative(oVariable, &bIsQualitative);

      for (iRow = 0; iRow < iNumberOfRows; ++iRow)
      {
         if (bIsQualitative == SQ_True)
         {
            /* Just set the first qualitative variable */
            SQ_GetQualitativeSettings(oVariable, &oStringVec);
            SQ_GetStringFromVector(oStringVec, 1, szBuffer, sizeof(szBuffer));
            *pszQualPos = (char*)strdup(szBuffer);
            ++pszQualPos;
         }
         else
         {
            /* Set a fake value */
            *pfQuantPos = ++fVal;
            ++pfQuantPos;
         }
      }
   }

   SQ_ClearPreparePrediction(&oPreparePrediction); /* This removes the variable and the variable vector object as well */
   SQ_ClearStringVector(&oStringVec);
}

void GetContributions(SQ_Model hModel)
{
   SQ_VectorData oData = 0;
   SQ_ErrorCode eErr;
   int iNumComponents = 0;

   SQ_GetNumberOfComponents(hModel, &iNumComponents);

   if (iNumComponents > 1)
   {
      /* Sample how to use the group contribution with multi weight. */
      /* We have more than one component, get the Score contributions with weight p1p2. Score Contrib(Obs {1, 2} - Average), Weight=p1p2 */
      SQ_IntVector oVec = 0;
      SQ_IntVector oWeightVec = 0;

      /* Observation number 1 and 2 AND same vector used for component 1 and 2.*/
      eErr = SQ_InitIntVector(&oVec, 2);
      eErr = SQ_SetDataInIntVector(oVec, 1, 1);
      eErr = SQ_SetDataInIntVector(oVec, 2, 2);

      /* Set the weights for the components = pp */
      eErr = SQ_InitIntVector(&oWeightVec, 2);
      eErr = SQ_SetDataInIntVector(oWeightVec, 1, SQ_Weight_P);
      eErr = SQ_SetDataInIntVector(oWeightVec, 2, SQ_Weight_P);

      eErr = SQ_GetContributionsScoresMultiWeightGroup(hModel, NULL/*average*/, &oVec/*observation 1 and 2*/, &oWeightVec /*weight pp*/, &oVec /*component 1 and 2*/, SQ_Reconstruct_Default, &oData);

      SQRunner_PrintVectorData(oData);

      SQ_ClearIntVector(&oVec);
      SQ_ClearIntVector(&oWeightVec);
      SQ_ClearVectorData(&oData);
   }
   else if (iNumComponents == 1)
   {
      /* Sample how to use the single observations contribution with single weight. */
      /* We only have one component, get the score contributions with weight p1. Score Contrib(Obs 1 - Average), Weight=p1 */
      eErr = SQ_GetContributionsScoresSingleWeight(hModel, 0/*average*/, 1/*observation 1*/, SQ_Weight_P, 1/*component 1*/, 0/*not used for weight p*/, SQ_Reconstruct_Default, &oData);

      SQRunner_PrintVectorData(oData);
      SQ_ClearVectorData(&oData);
   }
}

void ClearPredictionset(SQPredictionset* pSQPredSet)
{
   free(pSQPredSet->mpfQuantitativeValues);

   if (pSQPredSet->mpfQualitativeValues != NULL)
   {
      int i;
      int iNumItems = pSQPredSet->miNumQualCols * pSQPredSet->miNumRows;
      for (i = 0; i < iNumItems; ++i)
         free(pSQPredSet->mpfQualitativeValues[i]);

      free(pSQPredSet->mpfQualitativeValues);
   }
}
